﻿using Edu.Library;
using Edu.Library.Helpers;
using Edu.WinForms.EsyWin;
using EsyCode;
using Microsoft.CodeAnalysis.Scripting;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace Edu.WinForms
{
    public partial class FormEduIde : Form
    {
        protected readonly string BOOK_URL = @"https://www.librariacoresi.ro/shop/978-630-95008-3-0-matematica-si-informatica-in-c-5856";
        protected readonly string LARGE_SPACE = "     ";

        private bool _firstNew = true;
        protected IEduInOut EduInOut { get; set; }
        protected CancellationTokenSource CancelTokenSource { get; set; }


        public FormEduIde()
        {
            InitializeComponent();
        }

        private bool HasCode
        {
            get
            {
                return !string.IsNullOrEmpty(richTextBoxCode.Text.Trim());
            }
        }

        private bool ConfirmClearCode()
        {
            if (HasCode)
            {
                var result = Esy.ConfirmBox("Are you sure you want to clear the existing code?");
                return result;
            }
            return true;
        }

        private async void buttonRun_Click(object sender, EventArgs e)
        {
            try
            {
                if (!HasCode)
                {
                    Esy.MessageBox("No code to execute!");
                    return;
                }
                var code = richTextBoxCode.Text;
                CancelTokenSource = new CancellationTokenSource();
                var result = await CSharpCode.Execute(code, CancelTokenSource.Token);
            }
            catch (CompilationErrorException exCompil)
            {
                TreatCompilationError(exCompil);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "Runtime error!");
            }
        }

        private void TreatCompilationError(CompilationErrorException exCompil)
        {
            try
            {
                var diag = exCompil.Diagnostics.First();
                var line = diag.Location.GetLineSpan().StartLinePosition.Line;
                this.richTextBoxOutput.Text = $"line {line + 1}: {diag.GetMessage()}";
                var index = this.richTextBoxCode.GetFirstCharIndexFromLine(line);
                richTextBoxCode.Select(index, 3);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private async void FormEduIde_Load(object sender, EventArgs e)
        {
            try
            {
                this.Text = $"Esy Code v{Application.ProductVersion} - C# .Net";
                EduInOut = new EduInOut(pictureBoxScreen, richTextBoxOutput);
                Esy.InOut = EduInOut;
                LoadItems("Esy");
                // call Execute for performance reasons
                //var result = await CSharpCode.Execute("var x = 1;", new CancellationToken());
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }


        private void LoadItems(string typeName)
        {
            var typeNamespace = EsyNamespaces.GetNamespace(typeName);
            var dot = typeNamespace != "" ? "." : "";
            var fullTypeName = typeNamespace + dot + typeName;

            comboBoxItems.Items.Clear();
            comboBoxItems.Text = "";

            Assembly[] assembleis =
            {
                typeof(System.Drawing.Color).Assembly
            };

            var properties = ReflectionHelper
                .GetProperties(fullTypeName, assembleis)
                .OrderBy(pi => pi.Name)
                .Select(pi => pi.Name)
                .ToList();

            foreach (var property in properties)
            {
                comboBoxItems.Items.Add(property);
            }

            var methods = ReflectionHelper
                .GetMethods(fullTypeName, assembleis)
                .Where(
                    mi => mi.IsPublic &&
                    !mi.IsConstructor &&
                    !mi.IsSpecialName &&
                    !mi.IsVirtual
                )
                .OrderBy(mi => mi.Name)
                .ToList();

            foreach (var method in methods)
            {
                var txtParam = "";
                var parameters = method
                    .GetParameters()
                    .ToList();

                foreach (var param in parameters)
                {
                    var defValue = @"null";
                    if (param.ParameterType == typeof(string))
                    {
                        defValue = @"""""";
                    }
                    if (param.ParameterType == typeof(int) || param.ParameterType == typeof(float) || param.ParameterType == typeof(long) || param.ParameterType == typeof(double))
                    {
                        defValue = "0";
                    }
                    if (param.ParameterType == typeof(bool))
                    {
                        defValue = "false";
                    }
                    if (param.ParameterType == typeof(Color))
                    {
                        defValue = "Color.White";
                    }
                    txtParam += (txtParam == "" ? "" : ",") + defValue;
                }

                comboBoxItems.Items.Add($"{method.Name}({txtParam})");
            }
        }

        private void buttonClear_Click(object sender, EventArgs e)
        {
            try
            {
                EduInOut.Clear();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void richTextBoxCode_TextChanged(object sender, EventArgs e)
        {


        }

        private void buttonNew_Click(object sender, EventArgs e)
        {
            try
            {
                if (!ConfirmClearCode()) return;
                richTextBoxCode.Text = "";
                _firstNew = false;
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void buttonOpen_Click(object sender, EventArgs e)
        {
            try
            {
                var dialog = new OpenFileDialog();
                dialog.Filter = "All Files (*.*)|*.*|Rich Text Format (*.rtf)|*.rtf|Text (*.txt)|*.txt";
                var result = dialog.ShowDialog();
                if (result == DialogResult.Cancel) return;
                var extension = Path.GetExtension(dialog.FileName).ToLower();
                extension = extension.Replace(".", "");

                if (extension == "rtf")
                {
                    richTextBoxCode.LoadFile(dialog.FileName);
                    return;
                }

                var code = File.ReadAllText(dialog.FileName);
                richTextBoxCode.Text = code;
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void buttonSave_Click(object sender, EventArgs e)
        {
            try
            {
                var dialog = new SaveFileDialog();
                dialog.Filter = "Rich Text Format (*.rtf)|*.rtf";
                var result = dialog.ShowDialog();
                if (result == DialogResult.Cancel) return;
                richTextBoxCode.SaveFile(dialog.FileName);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void buttonClose_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void FormEduIde_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = !Esy.ConfirmBox("Are you sure you want to close the application?");
        }

        private void buttonCancel_Click(object sender, EventArgs e)
        {
            try
            {
                if (CancelTokenSource == null) return;
                CancelTokenSource.Cancel();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void buttonCopy_Click(object sender, EventArgs e)
        {
            try
            {
                var selectedText = richTextBoxCode.SelectedText;
                if (string.IsNullOrWhiteSpace(selectedText)) return;
                Clipboard.SetText(selectedText);
                richTextBoxCode.Select();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void buttonPaste_Click(object sender, EventArgs e)
        {
            try
            {
                richTextBoxCode.Paste();
                richTextBoxCode.Select();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void buttonUndo_Click(object sender, EventArgs e)
        {
            try
            {
                richTextBoxCode.Undo();
                richTextBoxCode.Select();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void buttonRedo_Click(object sender, EventArgs e)
        {
            try
            {
                richTextBoxCode.Redo();
                richTextBoxCode.Select();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void buttonSearch_Click(object sender, EventArgs e)
        {
            try
            {
                LoadItems(textBoxClass.Text.Trim());
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void buttonCopyItem_Click(object sender, EventArgs e)
        {
            try
            {
                var txt = textBoxClass.Text + "." + comboBoxItems.Text;
                Clipboard.SetText(txt);
                richTextBoxCode.Paste();
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.ToString());
            }
        }

        private void buttonSelectAll_Click(object sender, EventArgs e)
        {
            try
            {
                richTextBoxCode.Select();
                richTextBoxCode.SelectAll();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void buttonAbout_Click(object sender, EventArgs e)
        {

        }

        private void pictureBoxAbout_Click(object sender, EventArgs e)
        {
            var sbAbout = new StringBuilder();
            sbAbout.AppendLine($"Esy Code v{Application.ProductVersion}");
            sbAbout.AppendLine("Simple .Net Development Environment for learning C#");
            sbAbout.AppendLine($"C# Book: {BOOK_URL}");
            sbAbout.AppendLine($"{LARGE_SPACE}by Ioan-Valeriu Grossu (ioan.grossu@brahms.fizica.unibuc.ro)");
            sbAbout.AppendLine("Graphic design by Salma-Amalia El-Shamali:");
            sbAbout.AppendLine($@"{LARGE_SPACE}https://www.facebook.com/SalmaElShamal");
            sbAbout.AppendLine("Many thanks to: ");
            sbAbout.AppendLine($"{LARGE_SPACE}Igor Bustiuc (igor.bustiuc@gmail.com)");
            sbAbout.AppendLine($"{LARGE_SPACE}Teodor Tite (teotite@gmail.com)");
            sbAbout.AppendLine("for the support on cross-platform!");
            Esy.MessageBox(sbAbout.ToString());

            var sbLicense = new StringBuilder();
            sbLicense.AppendLine("MIT License\r\n\r\nCopyright (c) 2024 Ioan-Valeriu GROSSU\r\n\r\nPermission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the \"Software\"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:\r\n\r\nThe above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.\r\n\r\nTHE SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.");
            MessageBox.Show(sbLicense.ToString(), "License");
        }

        private void buttonUrl_Click(object sender, EventArgs e)
        {
            try
            {
                var success = WinUtils.TryOpenUrl(BOOK_URL);
 
                if (!success)
                {
                    Clipboard.SetText(BOOK_URL);
                    var sb = new StringBuilder();
                    sb.AppendLine($"The C# book URL has been copied to clipboard:");
                    sb.AppendLine(BOOK_URL);
                    Esy.MessageBox(sb.ToString());
                }
            }
            catch (Exception ex)
            {
                var sb = new StringBuilder();
                sb.AppendLine("C# Book:");
                sb.AppendLine(BOOK_URL);
                Esy.MessageBox(sb.ToString());
            }
        }
    }
}
